package mcfall.raytracer.tests;

import java.util.Arrays;
import java.util.List;

import mcfall.math.ColumnVector;
import mcfall.math.Matrix;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.math.Vector;
import mcfall.raytracer.objects.GenericCylinder;
import mcfall.raytracer.objects.GenericPlane;
import mcfall.raytracer.objects.GenericSphere;
import mcfall.raytracer.objects.HitRecord;

public class GenericObjects extends TestCase {

	double[] vectorComponents = new double[4];
	List<HitRecord> hitTimes;
	HitRecord[] hitArray = new HitRecord[2];
	
	/**
	 * Tests the intersection of a ray with the generic sphere.  Part of the example is 
	 * taken from Example 14.4.2 in Hill, Computer Graphics Using OpenGL, 2nd edition
	 *
	 */
	public void testGenericSphere () {
		GenericSphere sphere = new GenericSphere ();
		vectorComponents[0] = -3;
		vectorComponents[1] = -2;
		vectorComponents[2] = -3;
		vectorComponents[3] = 0;
		
		Ray ray = new Ray (new Point (3.0, 2.0, 3.0), new ColumnVector (4, vectorComponents));
		hitTimes = sphere.hitTime(ray);
		hitArray = hitTimes.toArray(hitArray);
		Arrays.sort(hitArray);
		assertAlmostEquals(0.7868, hitArray[0].hitTime);
		assertAlmostEquals(1.2132, hitArray[1].hitTime);
		
		//  Check the normal vectors at these points.  From Hill, the first hit point is
		//  (0.6393, 0.4264, 0.6396)
		assertAlmostEquals (0.6393, hitArray[0].normal.getValueAt(1));
		assertAlmostEquals (0.4264, hitArray[0].normal.getValueAt(2));
		assertAlmostEquals (0.6396, hitArray[0].normal.getValueAt(3));
		assertAlmostEquals (0.0, hitArray[0].normal.getValueAt(4));
		
		//  Due to the symmetry of the example, we can loop through the previous hit record to check the second one
		for (int i = 1; i <= 3; i++) {
			assertEquals (-1*hitArray[0].normal.getValueAt(i), hitArray[1].normal.getValueAt(i));
		}
		
		//  Test another example, with the eye at (2, 4, 6) and looking at the
		//  origin.  Hand-calculated answers are that the two hits should occur
		//  with t=1.13363 and t=0.86637
		vectorComponents[0] = -2.0;
		vectorComponents[1] = -4.0;
		vectorComponents[2] = -6.0;
		ray = new Ray (new Point(2, 4, 6), new ColumnVector (4, vectorComponents));
		hitTimes = sphere.hitTime(ray);
		hitArray = hitTimes.toArray(hitArray);
		Arrays.sort(hitArray);
		
		assertAlmostEquals (0.86637, hitArray[0].hitTime);
		assertAlmostEquals (1.13363, hitArray[1].hitTime);
		
		//  Now test a ray that should hit the sphere at exactly one point, the top
		vectorComponents[0] = -1.0;
		vectorComponents[1] = 0.0;
		vectorComponents[2] = 0.0;
		ray = new Ray (new Point(4, 1, 0), new ColumnVector(4, vectorComponents));
		hitTimes = sphere.hitTime(ray);	
		assertEquals (1,hitTimes.size());
		assertAlmostEquals(4.0, hitTimes.get(0).hitTime);
			
		//  Normal vector should be 0, 1, 0
		assertEquals (0.0, hitTimes.get(0).normal.getValueAt(1));
		assertEquals (1.0, hitTimes.get(0).normal.getValueAt(2));
		assertEquals (0.0, hitTimes.get(0).normal.getValueAt(3));
		assertEquals (0.0, hitTimes.get(0).normal.getValueAt(4));
		
		//  Finally try a ray that should miss the sphere altogether
		ray.setStart(new Point (4, 2, 0));
		hitTimes = sphere.hitTime(ray);
		assertEquals (0, hitTimes.size());		
		
		//  Now add a test that hits the bug we found when the sphere is rendered as an ellipse
		vectorComponents[0] = 0;
		vectorComponents[1] = -0.59999999;
		vectorComponents[2] = 1.5;
		ray = new Ray (new Point (0, 0, -3), new ColumnVector(4, vectorComponents));
		hitTimes = sphere.hitTime(ray);
		assertTrue (hitTimes.size() > 0);
	}
	
	public void testGenericSphereTransformed () {
		GenericSphere sphere = new GenericSphere ();
		sphere.transform(Matrix.createScalingMatrix(2, 1, 1));
		vectorComponents[0] = -1.0;
		vectorComponents[1] = 0.0;
		vectorComponents[2] = 0.0;
		vectorComponents[3] = 0.0;
		
		Ray ray = new Ray (new Point (4, 0, 0), new ColumnVector (4, vectorComponents));
		hitTimes = sphere.hitTime(ray);
		assertEquals (2, hitTimes.size());
		hitArray = hitTimes.toArray(hitArray);
		Arrays.sort(hitArray);
		assertAlmostEquals (2.0, hitArray[0].hitTime);
		
		//  Try a compound transformation, but translating the coordinate system to (4, 0, 0) and then scaling
		//  it by 2 in the x direction.  This should have the effect of placing the object's center at (4, 0) in the 
		//  original coordinate system, with the object's left boundary at (2,0) and the object's right boundary at (6, 0)
		//  in the original coordinate system.  Putting the ray at (8, 0, 0) moving to the left 1.0 gives a hit time of 2
		//  for the right boundary, and 6 for the left boundary
		sphere = new GenericSphere ();
		sphere.transform(Matrix.createTranslationMatrix(4, 0, 0));
		sphere.transform(Matrix.createScalingMatrix(2, 1, 1));
		ray.setStart(new Point (8, 0, 0));
		hitTimes = sphere.hitTime(ray);
		assertEquals (2, hitTimes.size());
		hitArray = hitTimes.toArray(hitArray);
		Arrays.sort(hitArray);
		assertAlmostEquals (2.0, hitArray[0].hitTime);
		assertAlmostEquals (6.0, hitArray[1].hitTime);
	}
	
	public void testGenericPlane () {		
		GenericPlane plane = new GenericPlane ();
		
		//  Try a ray that hits the plane at the origin
		vectorComponents[0] = 0;
		vectorComponents[1] = 0;
		vectorComponents[2] = -1;
		vectorComponents[3] = 0;
		
		Ray ray = new Ray (new Point (0, 0, 1), new ColumnVector (4, vectorComponents));
		hitTimes = plane.hitTime(ray);
		assertEquals (1, hitTimes.size());
		assertEquals (1.0, hitTimes.get(0).hitTime);
		Vector normalVector = hitTimes.get(0).normal;
		assertEquals (0.0, normalVector.getValueAt(1));
		assertEquals (0.0, normalVector.getValueAt(2));
		assertEquals (1.0, normalVector.getValueAt(3));
		assertEquals (0.0, normalVector.getValueAt(4));
		
		//  Now try a ray that is parallel to the plane
		vectorComponents[0] = 1;
		vectorComponents[1] = 0;
		vectorComponents[2] = 0;
		ray.setDirection(new ColumnVector(4, vectorComponents));
		hitTimes = plane.hitTime (ray);
		assertEquals (0, hitTimes.size());
		
	}
	
	public void testGenericCube () {
		
	}
	
	public void testGenericCylinder () {
		GenericCylinder cylinder = new GenericCylinder ();
		//  Create a ray that is below the cylinder in terms of y, and moves straight up.  It should hit the 
		//  cylinder at two points, one with y=-1 and one with y = 1
		vectorComponents[0] = 0;
		vectorComponents[1] = 1;
		vectorComponents[2] = 0;
		vectorComponents[3] = 0;
		
		Ray ray = new Ray (new Point (0, -2, 0.5), new ColumnVector(4, vectorComponents));
		hitTimes = cylinder.hitTime(ray);
		hitArray = hitTimes.toArray(hitArray);
		assertEquals (2, hitTimes.size());
		Arrays.sort(hitArray);
		assertEquals (1.0, hitArray[0].hitTime);
		assertEquals (3.0, hitArray[1].hitTime);
		
		assertEquals(0.0, hitArray[0].normal.getValueAt(1));
		assertEquals(-1.0, hitArray[0].normal.getValueAt(2));
		assertEquals(0.0, hitArray[0].normal.getValueAt(3));
		assertEquals(0.0, hitArray[0].normal.getValueAt(4));
		
		assertEquals(0.0, hitArray[1].normal.getValueAt(1));
		assertEquals(1.0, hitArray[1].normal.getValueAt(2));
		assertEquals(0.0, hitArray[1].normal.getValueAt(3));
		assertEquals(0.0, hitArray[1].normal.getValueAt(4));
		
		//  Try a ray that grazes the cylinder at just one point, at (0, 1, 0.5) by starting at (1, 1, 0.5) and moving to the left
		ray.setStart (new Point (1, 1, 0.5));
		vectorComponents[0] = -2;
		vectorComponents[1] = 0;
		vectorComponents[2] = 0;
		ray.setDirection(new ColumnVector(4, vectorComponents));
		hitTimes = cylinder.hitTime(ray);
		assertEquals(1, hitTimes.size());
		assertEquals (0.5, hitTimes.get(0).hitTime);
		
		assertEquals(0.0, hitTimes.get(0).normal.getValueAt(1));
		assertEquals(1.0, hitTimes.get(0).normal.getValueAt(2));
		assertEquals(0.0, hitTimes.get(0).normal.getValueAt(3));
		assertEquals(0.0, hitTimes.get(0).normal.getValueAt(4));
				
		//  Now try a ray that is below the cylinder, and runs parallel to it.  It shouldn't hit it at all
		vectorComponents[0] = 0;
		vectorComponents[1] = 0;
		vectorComponents[2] = 1;
		ray.setDirection(new ColumnVector(4, vectorComponents));
		hitTimes = cylinder.hitTime(ray);
		assertEquals (0, hitTimes.size());
		
		//  Try a ray that goes through the extended cylinder, but not the actual cylinder
		ray.setStart(new Point (0, -1, 2));
		vectorComponents[0] = 0;
		vectorComponents[1] = 1;
		vectorComponents[2] = 0;
		ray.setDirection(new ColumnVector(4, vectorComponents));
		hitTimes = cylinder.hitTime(ray);
		assertEquals (0, hitTimes.size());
		
		//  TODO:  Complete cylinder tests
		//  Try a ray that is to the right of the cylinder
		
		//  And to the left of it
	}
}
